/***************************************************************************//**
* \file SPIM_EZI2C.c
* \version 4.0
*
* \brief
*  This file provides the source code to the API for the SCB Component in
*  EZI2C mode.
*
* Note:
*
*******************************************************************************
* \copyright
* Copyright 2013-2017, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions,
* disclaimers, and limitations in the end user license agreement accompanying
* the software package with which this file was provided.
*******************************************************************************/

#include "SPIM_PVT.h"
#include "SPIM_EZI2C_PVT.h"


/***************************************
*      EZI2C Private Vars
***************************************/

volatile uint8 SPIM_curStatus; /* Status byte */
uint8 SPIM_fsmState;           /* FSM state   */

/* Variables intended to be used with Buffer 1: Primary slave address */
volatile uint8 * SPIM_dataBuffer1; /* Pointer to data buffer 1 */
uint16 SPIM_bufSizeBuf1;           /* Size of buffer 1 in bytes      */
uint16 SPIM_protectBuf1;           /* Start index of write protected area buffer 1 */
uint16 SPIM_offsetBuf1; /* Current offset within buffer 1 */
uint16 SPIM_indexBuf1;             /* Current index within buffer 1  */

#if(SPIM_SECONDARY_ADDRESS_ENABLE_CONST)
    uint8 SPIM_addrBuf1; /* Primary slave address. Used for software comparison   */
    uint8 SPIM_addrBuf2; /* Secondary slave address. Used for software comparison */

    /* Variables intended to be used with Buffer 2: Primary slave address */
    volatile uint8 * SPIM_dataBuffer2; /* Pointer to data buffer 2 */
    uint16 SPIM_bufSizeBuf2;           /* Size of buffer 2 in bytes */
    uint16 SPIM_protectBuf2;           /* Start index of write protected area buffer 2 */
    uint16 SPIM_offsetBuf2; /* Current offset within buffer 2 */
    uint16 SPIM_indexBuf2;             /* Current index within buffer 2 */
#endif /* (SPIM_SECONDARY_ADDRESS_ENABLE_CONST) */


/***************************************
*      EZI2C Private Functions
***************************************/

#if(SPIM_SECONDARY_ADDRESS_ENABLE_CONST)
    static uint32 SPIM_EzI2CUpdateRxMatchReg(uint32 addr1, uint32 addr2);
#endif /* (SPIM_SECONDARY_ADDRESS_ENABLE_CONST) */

#if(SPIM_SCB_MODE_UNCONFIG_CONST_CFG)

    /***************************************
    *  Configuration Structure Initialization
    ***************************************/

    /* Constant configuration of EZI2C */
    const SPIM_EZI2C_INIT_STRUCT SPIM_configEzI2C =
    {
        SPIM_EZI2C_CLOCK_STRETCHING,
        SPIM_EZI2C_MEDIAN_FILTER_ENABLE,
        SPIM_EZI2C_NUMBER_OF_ADDRESSES,
        SPIM_EZI2C_PRIMARY_SLAVE_ADDRESS,
        SPIM_EZI2C_SECONDARY_SLAVE_ADDRESS,
        SPIM_EZI2C_SUB_ADDRESS_SIZE,
        SPIM_EZI2C_WAKE_ENABLE,
        SPIM_EZI2C_BYTE_MODE_ENABLE
    };

    /*******************************************************************************
    * Function Name: SPIM_EzI2CInit
    ****************************************************************************//**
    *
    *  Configures the SPIM for EZI2C operation.
    *
    *  This function is intended specifically to be used when the SPIM 
    *  configuration is set to “Unconfigured SPIM” in the customizer. 
    *  After initializing the SPIM in EZI2C mode using this function, 
    *  the component can be enabled using the SPIM_Start() or 
    * SPIM_Enable() function.
    *  This function uses a pointer to a structure that provides the configuration 
    *  settings. This structure contains the same information that would otherwise 
    *  be provided by the customizer settings.
    *
    * \param config: pointer to a structure that contains the following list of 
    *  fields. These fields match the selections available in the customizer. 
    *  Refer to the customizer for further description of the settings.
    *
    *******************************************************************************/
    void SPIM_EzI2CInit(const SPIM_EZI2C_INIT_STRUCT *config)
    {
        if(NULL == config)
        {
            CYASSERT(0u != 0u); /* Halt execution due to bad function parameter */
        }
        else
        {
            /* Configure pins */
            SPIM_SetPins(SPIM_SCB_MODE_EZI2C, SPIM_DUMMY_PARAM,
                                                                      SPIM_DUMMY_PARAM);

            /* Store internal configuration */
            SPIM_scbMode       = (uint8) SPIM_SCB_MODE_EZI2C;
            SPIM_scbEnableWake = (uint8) config->enableWake;
            SPIM_scbEnableIntr = (uint8) SPIM_SCB_IRQ_INTERNAL;

            SPIM_numberOfAddr  = (uint8) config->numberOfAddresses;
            SPIM_subAddrSize   = (uint8) config->subAddrSize;

        #if (SPIM_CY_SCBIP_V0)
            /* Adjust SDA filter settings. Ticket ID#150521 */
            SPIM_SET_I2C_CFG_SDA_FILT_TRIM(SPIM_EC_AM_I2C_CFG_SDA_FILT_TRIM);
        #endif /* (SPIM_CY_SCBIP_V0) */

            /* Adjust AF and DF filter settings, AF = 1, DF = 0. Ticket ID#176179 */
            SPIM_I2C_CFG_ANALOG_FITER_ENABLE;

            /* Configure I2C interface */
            SPIM_CTRL_REG     = SPIM_GET_CTRL_BYTE_MODE  (config->enableByteMode)    |
                                            SPIM_GET_CTRL_ADDR_ACCEPT(config->numberOfAddresses) |
                                            SPIM_GET_CTRL_EC_AM_MODE (config->enableWake);

            SPIM_I2C_CTRL_REG = SPIM_EZI2C_CTRL;

            /* Configure RX direction */
            SPIM_RX_CTRL_REG = SPIM_EZI2C_RX_CTRL |
                                        SPIM_GET_RX_CTRL_MEDIAN(SPIM_DIGITAL_FILTER_DISABLE);
                                                ;
            SPIM_RX_FIFO_CTRL_REG = SPIM_CLEAR_REG;

            /* Set default address and mask */
            if(SPIM_EZI2C_PRIMARY_ADDRESS == config->numberOfAddresses)
            {
                SPIM_RX_MATCH_REG = SPIM_EzI2CUpdateRxMatchReg(config->primarySlaveAddr,
                                                                                       config->primarySlaveAddr);
            }
            else
            {
                SPIM_RX_MATCH_REG = SPIM_EzI2CUpdateRxMatchReg(config->primarySlaveAddr,
                                                                                       config->secondarySlaveAddr);
            }

            /* Configure TX direction */
            SPIM_TX_CTRL_REG      = SPIM_EZI2C_TX_CTRL;
            SPIM_TX_FIFO_CTRL_REG = ((0u != (config->enableClockStretch)) ?
                                                 SPIM_CLEAR_REG : SPIM_EZI2C_TX_FIFO_CTRL);

            /* Configure interrupt sources */
        #if (!SPIM_CY_SCBIP_V1)
           SPIM_INTR_SPI_EC_MASK_REG = SPIM_NO_INTR_SOURCES;
        #endif /* (!SPIM_CY_SCBIP_V1) */

            SPIM_INTR_I2C_EC_MASK_REG = SPIM_NO_INTR_SOURCES;
            SPIM_INTR_MASTER_MASK_REG = SPIM_NO_INTR_SOURCES;
            SPIM_INTR_SLAVE_MASK_REG  = SPIM_EZI2C_INTR_SLAVE_MASK;
            SPIM_INTR_TX_MASK_REG     = SPIM_NO_INTR_SOURCES;

            /* Configure interrupt with EZI2C handler but do not enable it */
            CyIntDisable    (SPIM_ISR_NUMBER);
            CyIntSetPriority(SPIM_ISR_NUMBER, SPIM_ISR_PRIORITY);
            (void) CyIntSetVector(SPIM_ISR_NUMBER, (0u != (config->enableClockStretch)) ?
                                                                      (&SPIM_EZI2C_STRETCH_ISR) :
                                                                      (&SPIM_EZI2C_NO_STRETCH_ISR));

            if(0u != (config->enableClockStretch))
            {
                /* Configure interrupt sources */
                SPIM_INTR_SLAVE_MASK_REG |= SPIM_INTR_SLAVE_I2C_ADDR_MATCH;
                SPIM_INTR_RX_MASK_REG     = SPIM_NO_INTR_SOURCES;
            }
            else
            {
                /* Enable Auto ACK/NACK features */
                SPIM_I2C_CTRL_REG |= SPIM_EZI2C_CTRL_AUTO;

                /* Configure RX interrupt source */
                SPIM_INTR_SLAVE_MASK_REG |= SPIM_INTR_SLAVE_I2C_START;
                SPIM_INTR_RX_MASK_REG     = SPIM_INTR_RX_NOT_EMPTY;
            }

            /* Configure global variables */
            SPIM_fsmState = SPIM_EZI2C_FSM_IDLE;

            SPIM_curStatus  = 0u;
            SPIM_indexBuf1  = 0u;
            SPIM_offsetBuf1 = 0u;
            SPIM_indexBuf2  = 0u;
            SPIM_offsetBuf2 = 0u;

            SPIM_addrBuf1 = (uint8) config->primarySlaveAddr;
            SPIM_addrBuf2 = (uint8) config->secondarySlaveAddr;
        }
    }

#else

    /*******************************************************************************
    * Function Name: SPIM_EzI2CInit
    ****************************************************************************//**
    *
    *  Configures the SCB for the EZI2C operation.
    *
    *******************************************************************************/
    void SPIM_EzI2CInit(void)
    {
        /* Configure I2C interface */
        SPIM_CTRL_REG     = SPIM_EZI2C_DEFAULT_CTRL;
        SPIM_I2C_CTRL_REG = SPIM_EZI2C_DEFAULT_I2C_CTRL;

    #if (SPIM_CY_SCBIP_V0)
        /* Adjust SDA filter settings. Ticket ID#150521 */
        SPIM_SET_I2C_CFG_SDA_FILT_TRIM(SPIM_EC_AM_I2C_CFG_SDA_FILT_TRIM);
    #endif /* (SPIM_CY_SCBIP_V0) */

        /* Configure RX direction */
        SPIM_RX_CTRL_REG      = SPIM_EZI2C_DEFAULT_RX_CTRL;
        SPIM_RX_FIFO_CTRL_REG = SPIM_EZI2C_DEFAULT_RX_FIFO_CTRL;

        /* Set default address and mask */
        SPIM_RX_MATCH_REG     = SPIM_EZI2C_DEFAULT_RX_MATCH;

        /* Configure TX direction */
        SPIM_TX_CTRL_REG      = SPIM_EZI2C_DEFAULT_TX_CTRL;
        SPIM_TX_FIFO_CTRL_REG = SPIM_EZI2C_DEFAULT_TX_FIFO_CTRL;

        /* Configure interrupt with EZI2C handler but do not enable it */
    #if !defined (CY_EXTERNAL_INTERRUPT_CONFIG)
        CyIntDisable    (SPIM_ISR_NUMBER);
        CyIntSetPriority(SPIM_ISR_NUMBER, SPIM_ISR_PRIORITY);

    #if (SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST)
        (void) CyIntSetVector(SPIM_ISR_NUMBER, &SPIM_EZI2C_STRETCH_ISR);
    #else
        (void) CyIntSetVector(SPIM_ISR_NUMBER, &SPIM_EZI2C_NO_STRETCH_ISR);
    #endif /* (SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST) */

    #endif /* !defined (CY_EXTERNAL_INTERRUPT_CONFIG) */

        /* Configure interrupt sources */
    #if (!SPIM_CY_SCBIP_V1)
        SPIM_INTR_SPI_EC_MASK_REG = SPIM_EZI2C_DEFAULT_INTR_SPI_EC_MASK;
    #endif /* (!SPIM_CY_SCBIP_V1) */

        SPIM_INTR_I2C_EC_MASK_REG = SPIM_EZI2C_DEFAULT_INTR_I2C_EC_MASK;
        SPIM_INTR_SLAVE_MASK_REG  = SPIM_EZI2C_DEFAULT_INTR_SLAVE_MASK;
        SPIM_INTR_MASTER_MASK_REG = SPIM_EZI2C_DEFAULT_INTR_MASTER_MASK;
        SPIM_INTR_RX_MASK_REG     = SPIM_EZI2C_DEFAULT_INTR_RX_MASK;
        SPIM_INTR_TX_MASK_REG     = SPIM_EZI2C_DEFAULT_INTR_TX_MASK;

        /* Configure global variables */
        SPIM_fsmState = SPIM_EZI2C_FSM_IDLE;

        SPIM_curStatus  = 0u;
        SPIM_indexBuf1  = 0u;
        SPIM_offsetBuf1 = 0u;

    #if(SPIM_SECONDARY_ADDRESS_ENABLE_CONST)
        SPIM_indexBuf2  = 0u;
        SPIM_offsetBuf2 = 0u;

        SPIM_addrBuf1 = SPIM_EZI2C_PRIMARY_SLAVE_ADDRESS;
        SPIM_addrBuf2 = SPIM_EZI2C_SECONDARY_SLAVE_ADDRESS;
    #endif /* (SPIM_SECONDARY_ADDRESS_ENABLE_CONST) */
    }
#endif /* (SPIM_SCB_MODE_UNCONFIG_CONST_CFG) */


/*******************************************************************************
* Function Name: SPIM_EzI2CStop
****************************************************************************//**
*
*  Resets the EZI2C FSM into the default state.
*
*******************************************************************************/
void SPIM_EzI2CStop(void)
{
    SPIM_fsmState = SPIM_EZI2C_FSM_IDLE;
}


/*******************************************************************************
* Function Name: SPIM_EzI2CGetActivity
****************************************************************************//**
*
*  Returns the EZI2C slave status.
*  The read, write and error status flags reset to zero after this function
*  call. The busy status flag is cleared when the transaction intended for
*  the EZI2C slave completes.
*
* \return
*  Returns the status of the EZI2C Slave activity.
*  - SPIM_EZI2C_STATUS_READ1 - Read transfer complete. The transfer 
*    used the primary slave address. The error condition status bit must be 
*    checked to ensure that read transfer was completed successfully.
*  - SPIM_EZI2C_STATUS_WRITE1 - Write transfer complete. The buffer 
*    content was modified. The transfer used the primary slave address. 
*    The error condition status bit must be checked to ensure that write 
*    transfer was completed successfully.
*  - SPIM_EZI2C_STATUS_READ2 - Read transfer complete. The transfer
*    used the secondary slave address. The error condition status bit must be 
*    checked to ensure that read transfer was completed successfully.
*  - SPIM_EZI2C_STATUS_WRITE2 - Write transfer complete. The buffer
*    content was modified. The transfer used the secondary slave address. 
*    The error condition status bit must be checked to ensure that write 
*    transfer was completed successfully.
*  - SPIM_EZI2C_STATUS_BUSY - A transfer intended for the primary 
*    or secondary address is in progress. The status bit is set after an 
*    address match and cleared on a Stop or ReStart condition.
*  - SPIM_EZI2C_STATUS_ERR - An error occurred during a transfer 
*    intended for the primary or secondary slave address. The sources of error
*    are: misplaced Start or Stop condition or lost arbitration while slave 
*    drives SDA.
*    The write buffer may contain invalid byte or part of the transaction when 
*    SPIM_EZI2C_STATUS_ERR and SPIM_EZI2C_STATUS_WRITE1/2 
*    is set. It is recommended to discard buffer content in this case.
*
* \globalvars
*  SPIM_curStatus - used to store the current status of the EZI2C 
*  slave.
*
*******************************************************************************/
uint32 SPIM_EzI2CGetActivity(void)
{
    uint32 status;

    SPIM_DisableInt();  /* Lock from interruption */

    status = SPIM_curStatus;

    /* Relay on address match event from HW as bus busy status */
    if(SPIM_EZI2C_SCL_STRETCH_DISABLE)
    {
        /* For SPIM_CY_SCBIP_V0 the wake is prohibited by customizer */
        if(SPIM_EZI2C_EC_AM_ENABLE)
        {
            status |= SPIM_CHECK_INTR_I2C_EC(SPIM_INTR_I2C_EC_WAKE_UP) ?
                        SPIM_EZI2C_STATUS_BUSY : 0u;
        }
        else
        {
            status |= SPIM_CHECK_INTR_SLAVE(SPIM_INTR_SLAVE_I2C_ADDR_MATCH) ?
                        SPIM_EZI2C_STATUS_BUSY : 0u;
        }
        
    }
    

    SPIM_curStatus &= ((uint8) ~SPIM_EZI2C_CLEAR_STATUS);

    SPIM_EnableInt();   /* Release lock */

    return(status);
}


/*******************************************************************************
* Function Name: SPIM_EzI2CSetAddress1
****************************************************************************//**
*
*  Sets the primary EZI2C slave address.
*
* \param address: I2C slave address for the secondary device.
*  This address is the 7-bit right-justified slave address and does not 
*   include the R/W bit.
*
*  The address value is not checked to see if it violates the I2C spec. 
*  The preferred addresses are in the range between 8 and 120 (0x08 to 0x78).
*
* \globalvars
*  SPIM_addrBuf1 - used to store the primary 7-bit slave address 
*  value.
*
*******************************************************************************/
void SPIM_EzI2CSetAddress1(uint32 address)
{
    if(SPIM_SECONDARY_ADDRESS_ENABLE)
    {
        SPIM_addrBuf1 = (uint8) address;

        SPIM_RX_MATCH_REG = 
                        SPIM_EzI2CUpdateRxMatchReg(address, (uint32) SPIM_addrBuf2);
    }
    else
    {
        uint32 matchReg;

        matchReg = SPIM_RX_MATCH_REG;

        /* Set address. */
        matchReg &= (uint32) ~SPIM_RX_MATCH_ADDR_MASK;
        matchReg |= (uint32)  SPIM_GET_I2C_8BIT_ADDRESS(address);

        SPIM_RX_MATCH_REG = matchReg;
    }
    
}


/*******************************************************************************
* Function Name: SPIM_EzI2CGetAddress1
****************************************************************************//**
*
*  Returns primary the EZ I2C slave address.
*  This address is the 7-bit right-justified slave address and does not include 
*  the R/W bit.
*
* \return
*  Primary EZI2C slave address.
*
* \globalvars
*  SPIM_addrBuf1 - used to store the primary 7-bit slave address 
*  value.
*
*******************************************************************************/
uint32 SPIM_EzI2CGetAddress1(void)
{
    uint32 address;

    if(SPIM_SECONDARY_ADDRESS_ENABLE)
    {
        address = (uint32) SPIM_addrBuf1;
    }
    else
    {
        address = (SPIM_GET_RX_MATCH_ADDR(SPIM_RX_MATCH_REG) >>
                   SPIM_I2C_SLAVE_ADDR_POS);
    }
    

    return(address);
}


/*******************************************************************************
* Function Name: SPIM_EzI2CSetBuffer1
****************************************************************************//**
*
*  Sets up the data buffer to be exposed to the I2C master on a primary slave
*  address request.
*
* \param bufSize: Size of the buffer in bytes.
* \param rwBoundary: Number of data bytes starting from the beginning of the 
*  buffer with read and write access. Data bytes located at offset rwBoundary 
*  or greater are read only.
*  This value must be less than or equal to the buffer size.
* \param buffer: Pointer to the data buffer.
*
* \sideeffect
*  Calling this function in the middle of a transaction intended for the 
*  primary slave address leads to unexpected behavior.
*
* \globalvars
*  SPIM_dataBuffer1 – the pointer to the buffer to be exposed to the
*  master on a primary address.
*  SPIM_bufSizeBuf1 - the size of the buffer to be exposed to the 
*  master on a primary address.
*  SPIM_protectBuf1 - the start index of the read-only region in the
*  buffer to be exposed to the master on a primary address. The read-only region
*  continues up to the end the buffer.
*
*******************************************************************************/
void SPIM_EzI2CSetBuffer1(uint32 bufSize, uint32 rwBoundary, volatile uint8 * buffer)
{
    if (NULL != buffer)
    {
        SPIM_DisableInt();  /* Lock from interruption */

        SPIM_dataBuffer1 =  buffer;
        SPIM_bufSizeBuf1 = (uint16) bufSize;
        SPIM_protectBuf1 = (uint16) rwBoundary;

        SPIM_EnableInt();   /* Release lock */
    }
}


/*******************************************************************************
* Function Name: SPIM_EzI2CSetReadBoundaryBuffer1
****************************************************************************//**
*
*  Sets the read only boundary in the data buffer to be exposed to the I2C
*  master on a primary slave address request.
*
* \param rwBoundry: Number of data bytes starting from the beginning of the 
*  buffer with read and write access. Data bytes located at offset rwBoundary 
*  or greater are read only.
*  This value must be less than or equal to the buffer size.
*
* \sideeffect
*  Calling this function in the middle of a transaction intended for the 
*  primary slave address leads to unexpected behavior.
*
* \globalvars
*  SPIM_protectBuf1 - the start index of the read-only region in the
*  buffer to be exposed to the master on a primary address. The read-only region
*  continues up to the end the buffer.
*
*******************************************************************************/
void SPIM_EzI2CSetReadBoundaryBuffer1(uint32 rwBoundary)
{
    SPIM_protectBuf1 = (uint16) rwBoundary;
}


#if(SPIM_SECONDARY_ADDRESS_ENABLE_CONST)
    /*******************************************************************************
    * Function Name: SPIM_EzI2CUpdateRxMatchReg
    ****************************************************************************//**
    *
    *  Returns the value of the RX MATCH register for addr1 and addr2. The addr1 is
    *  accepted as the primary address and it is written to RX_MATCH.ADDRESS
    *  (addr1 << 0x01).
    *  The RX_MATCH.MASK is set as follow: addr1 and addr2 equal bits set to 1
    *  otherwise 0.
    *
    * \param addr1: I2C slave address for the primary device.
    * \param addr2: I2C slave address for the secondary device.
    *  This address is the 7-bit right-justified slave address and does
    *  not include the R/W bit.
    *
    * \return
    *  Value of RX MATCH register.
    *
    *******************************************************************************/
    static uint32 SPIM_EzI2CUpdateRxMatchReg(uint32 addr1, uint32 addr2)
    {
        uint32 matchReg;

        matchReg  = ~(addr1 ^ addr2); /* If (addr1 and addr2) bit matches - mask bit equals 1, in other case 0 */

        matchReg  = (uint32) (SPIM_GET_I2C_8BIT_ADDRESS(matchReg) << SPIM_RX_MATCH_MASK_POS);
        matchReg |= SPIM_GET_I2C_8BIT_ADDRESS(addr1);

        return(matchReg);
    }

    /*******************************************************************************
    * Function Name: SPIM_EzI2CSetAddress2
    ****************************************************************************//**
    *
    *  Sets the secondary EZI2C slave address.
    *
    * \param address: secondary I2C slave address.
    *  This address is the 7-bit right-justified slave address and does not 
    *  include the R/W bit.
    *  The address value is not checked to see if it violates the I2C spec. 
    *  The preferred addresses are in the range between 8 and 120 (0x08 to 0x78).
    *
    * \globalvars
    *  SPIM_addrBuf2 - used to store the secondary 7-bit slave address 
    *  value.
    *
    *******************************************************************************/
    void SPIM_EzI2CSetAddress2(uint32 address)
    {
        SPIM_addrBuf2 = (uint8) address;

        SPIM_RX_MATCH_REG = 
                        SPIM_EzI2CUpdateRxMatchReg((uint32) SPIM_addrBuf1, address);
    }


    /*******************************************************************************
    * Function Name: SPIM_EzI2CGetAddress2
    ****************************************************************************//**
    *
    *  Returns the secondary EZ I2C slave address.
    *  This address is the 7-bit right-justified slave address and does not include 
    *  the R/W bit.
    *
    * \return
    *  Secondary I2C slave address.
    *
    * \globalvars
    *  SPIM_addrBuf2 - used to store the secondary 7-bit slave address 
    *  value.
    *
    *******************************************************************************/
    uint32 SPIM_EzI2CGetAddress2(void)
    {
        return((uint32) SPIM_addrBuf2);
    }


    /*******************************************************************************
    * Function Name: SPIM_EzI2CSetBuffer2
    ****************************************************************************//**
    *
    *  Sets up the data buffer to be exposed to the I2C master on a secondary slave
    *  address request.
    *
    * \param bufSize: Size of the buffer in bytes.
    * \param rwBoundary: Number of data bytes starting from the beginning of the 
    *  buffer with read and write access. Data bytes located at offset rwBoundary 
    *  or greater are read only.
    *  This value must be less than or equal to the buffer size.
    * \param buffer: Pointer to the data buffer.
    *
    * \sideeffects
    *  Calling this function in the middle of a transaction intended for the 
    *  secondary slave address leads to unexpected behavior.
    *
    * \globalvars
    *  SPIM_dataBuffer2 – the pointer to the buffer to be exposed to the
    *  master on a secondary address.
    *  SPIM_bufSizeBuf2 - the size of the buffer to be exposed to the 
    *  master on a secondary address.
    *  SPIM_protectBuf2 - the start index of the read-only region in the
    *  buffer to be exposed to the master on a secondary address. The read-only 
    *  region continues up to the end the buffer.
    *
    *******************************************************************************/
    void SPIM_EzI2CSetBuffer2(uint32 bufSize, uint32 rwBoundary, volatile uint8 * buffer)
    {
        if (NULL != buffer)
        {
            SPIM_DisableInt();  /* Lock from interruption */

            SPIM_dataBuffer2 =  buffer;
            SPIM_bufSizeBuf2 = (uint16) bufSize;
            SPIM_protectBuf2 = (uint16) rwBoundary;

            SPIM_EnableInt();   /* Release lock */
        }
    }


    /*******************************************************************************
    * Function Name: SPIM_EzI2CSetReadBoundaryBuffer2
    ****************************************************************************//**
    *
    *  Sets the read only boundary in the data buffer to be exposed to the I2C
    *  master on a secondary address request.
    *
    *  \param rwBoundary: Number of data bytes starting from the beginning of the
    *   buffer with read and write access. Data bytes located at offset rwBoundary 
    *   or greater are read only.
    *   This value must be less than or equal to the buffer size.
    *
    *  \sideeffect
    *   Calling this function in the middle of a transaction intended for the 
    *   secondary slave address leads to unexpected behavior.
    *
    * \globalvars
    *  SPIM_protectBuf2 - the start index of the read-only region in the
    *  buffer to be exposed to the master on a secondary address. The read-only 
    *  region continues up to the end the buffe
    *
    *******************************************************************************/
    void SPIM_EzI2CSetReadBoundaryBuffer2(uint32 rwBoundary)
    {
        SPIM_protectBuf2 = (uint16) rwBoundary;
    }

#endif /* (SPIM_SECONDARY_ADDRESS_ENABLE_CONST) */


#if(SPIM_EZI2C_WAKE_ENABLE_CONST)
    /*******************************************************************************
    * Function Name: SPIM_EzI2CSaveConfig
    ****************************************************************************//**
    *
    *  Clock stretching is  enabled: Enables INTR_I2C_EC.WAKE_UP interrupt source.
    *  It triggers on the slave address match.
    *  Clock stretching is disabled: Waits until the I2C slave becomes free and
    *  disables the block to perform reconfiguration from the active mode operation
    *  to deep sleep with wake up on the address match: enables INTR_I2C_EC.WAKE_UP
    *  interrupt source and disables the INTR_S and INTR_TX interrupt sources.
    *  The block is disabled before reconfiguration and enabled when
    *  it is completed.
    *
    *******************************************************************************/
    void SPIM_EzI2CSaveConfig(void)
    {
    #if(SPIM_CY_SCBIP_V0)

        if(SPIM_EZI2C_SCL_STRETCH_ENABLE)
        {
            /* Enable wakeup interrupt source on address match */
            SPIM_SetI2CExtClkInterruptMode(SPIM_INTR_I2C_EC_WAKE_UP);
        }
        

    #else
        uint8 enableInt;

        enableInt = (uint8) SPIM_INTR_I2C_EC_WAKE_UP;

        if(SPIM_EZI2C_SCL_STRETCH_ENABLE)
        {
        #if (SPIM_SCB_CLK_INTERNAL)
            /* Disable clock to internal address match logic. Ticket ID #187931 */
            SPIM_SCBCLK_Stop();
        #endif /* (SPIM_SCB_CLK_INTERNAL) */

            /* Enable interrupt source to wakeup device */
            SPIM_SetI2CExtClkInterruptMode(enableInt);
        }
        else
        {
            for(;;) /* Wait for end of transaction intended to slave */
            {
                if(0u == (SPIM_GetI2CExtClkInterruptSource() & SPIM_INTR_I2C_EC_WAKE_UP))
                {
                    enableInt = CyEnterCriticalSection();

                    if(0u == (SPIM_GetI2CExtClkInterruptSource() & SPIM_INTR_I2C_EC_WAKE_UP))
                    {
                        /* Attempts to set NACK command before disable block */
                        SPIM_I2C_SLAVE_GENERATE_NACK;

                        if(0u == (SPIM_GetI2CExtClkInterruptSource() & SPIM_INTR_I2C_EC_WAKE_UP))
                        {
                            /* NACK command was set before. It is safe to disable block */
                            SPIM_CTRL_REG &= (uint32) ~SPIM_CTRL_ENABLED;
                            SPIM_DisableInt();

                            CyExitCriticalSection(enableInt);
                            break;
                        }
                        else
                        {
                            /* Clear NACK command to prevent data NACK */
                            SPIM_I2C_SLAVE_CLEAR_NACK;
                        }
                    }

                    CyExitCriticalSection(enableInt);
                }
            }

            /* Disable all active mode interrupt sources */
            SPIM_SetTxInterruptMode(SPIM_NO_INTR_SOURCES);
            SPIM_SetSlaveInterruptMode(SPIM_NO_INTR_SOURCES);
            SPIM_ClearPendingInt();
            SPIM_EnableInt();

            /* Enable wakeup interrupt on address match */
            SPIM_SetI2CExtClkInterruptMode(SPIM_INTR_I2C_EC_WAKE_UP);

            enableInt = CyEnterCriticalSection();

            SPIM_CTRL_REG |= (uint32) SPIM_CTRL_ENABLED;
            SPIM_I2C_SLAVE_GENERATE_NACK;

            CyExitCriticalSection(enableInt);
        }
        
    #endif /* (SPIM_CY_SCBIP_V0) */
    }


    /*******************************************************************************
    * Function Name: SPIM_EzI2CRestoreConfig
    ****************************************************************************//**
    *
    *  Clock stretching is  enabled: Disables the INTR_I2C_EC.WAKE_UP interrupt
    *  source.
    *  Clock stretching is disabled: Reconfigures the EZI2C component from
    *  Deep Sleep (wake up on the address match) to active operation: disables
    *  the INTR_I2C_EC.WAKE_UP interrupt source and restores the INTR_S
    *  interrupt sources to operate in the active mode.
    *  The block is disabled before reconfiguration and enabled when
    *  it is completed.
    *
    *******************************************************************************/
    void SPIM_EzI2CRestoreConfig(void)
    {
    #if(SPIM_CY_SCBIP_V0)

        if(SPIM_EZI2C_SCL_STRETCH_ENABLE)
        {
            /* Disable wakeup interrupt on address match */
            SPIM_SetI2CExtClkInterruptMode(SPIM_NO_INTR_SOURCES);
        }
        

    #else

        if(SPIM_EZI2C_SCL_STRETCH_ENABLE)
        {
            /* Disable wakeup interrupt source on address match */
            SPIM_SetI2CExtClkInterruptMode(SPIM_NO_INTR_SOURCES);

        #if (SPIM_SCB_CLK_INTERNAL)
            /* Enable clock to internal address match logic. Ticket ID #187931 */
            SPIM_SCBCLK_Start();
        #endif /* (SPIM_SCB_CLK_INTERNAL) */
        }
        else
        {
            /* NACK will be driven on the bus by wakeup or NACK command.
            * It is safe to disable block to restore active mode configuration.
            */
            SPIM_CTRL_REG &= (uint32) ~SPIM_CTRL_ENABLED;

            /* Restore active mode interrupt sources */
            SPIM_SetI2CExtClkInterruptMode(SPIM_NO_INTR_SOURCES);
            SPIM_SetSlaveInterruptMode(SPIM_EZI2C_INTR_SLAVE_MASK |
                                                   SPIM_INTR_SLAVE_I2C_START);
            SPIM_ClearPendingInt();

            SPIM_CTRL_REG |= (uint32) SPIM_CTRL_ENABLED;
        }
        

    #endif /* (SPIM_CY_SCBIP_V0) */
    }
#endif /* (SPIM_EZI2C_WAKE_ENABLE_CONST) */


/* [] END OF FILE */
